/*******************************************************************************
NAME                           INV_INIT 

PURPOSE:	Initializes inverse projection transformation parameters

ALGORITHM REFERENCES

1.  Snyder, John P., "Map Projections--A Working Manual", U.S. Geological
    Survey Professional Paper 1395 (Supersedes USGS Bulletin 1532), United
    State Government Printing Office, Washington D.C., 1987.

2.  Snyder, John P. and Voxland, Philip M., "An Album of Map Projections",
    U.S. Geological Survey Professional Paper 1453 , United State Government
    Printing Office, Washington D.C., 1989.
*******************************************************************************/
#include "cproj.h"
#include "gctp.h"

void inv_init
(
    long insys,		/* input system code				*/
    long inzone,		/* input zone number				*/
    const double *inparm, /* input array of projection parameters		*/
    long inspheroid,	/* input spheroid code				*/
    long *iflg,		/* status flag					*/
    long (*inv_trans[])(double,double,double*,double*)	
                    /* inverse function pointer			*/
)
{
double angle;		/* rotation anlge				*/
double lat1 = 0.0;		/* 1st standard parallel			*/
double lat2 = 0.0;		/* 2nd standard parallel			*/
double center_long;	/* center longitude				*/
double center_lat;	/* center latitude				*/
double h;		/* height above sphere				*/
double lat_origin;	/* latitude at origin				*/
double r_major;		/* major axis in meters				*/
double r_minor;		/* minor axis in meters				*/
double false_easting;	/* false easting in meters			*/
double false_northing;	/* false northing in meters			*/
double radius;		/* radius of sphere				*/
double shape_m;		/* constant used for Oblated Equal Area		*/
double shape_n;		/* constant used for Oblated Equal Area		*/
long mode;		/* which format is used	A or B			*/
double dzone;       /* number of longitudinal zones in ISG          */
double djustify;    /* justify flag in ISG projection               */

	/* Function declarations for inverse function pointer
	---------------------------------------------------*/
long alberinv();
long merinv();
long eqconinv();
long sterinv();
long lamazinv();
long aziminv();
long gnominv();
long orthinv();
long gvnspinv();
long sininv();
long equiinv();
long millinv();
long vandginv();
long haminv();
long robinv();
long goodinv();
long molwinv();
long imolwinv();
long alconinv();
long wivinv();
long wviiinv();
long obleqinv();
long isinusinv();

/* Initialize inverse transformations
-----------------------------------*/
  /* find the correct major and minor axis
  --------------------------------------*/
  sphdz(inspheroid,inparm,&r_major,&r_minor,&radius);
  false_easting  = inparm[6];
  false_northing = inparm[7];

  if ((insys == HOM)
      || (insys == LAMCC)
      || (insys == UTM)
      || (insys == TM)
      || (insys == POLYC)
      || (insys == PS)
      || (insys == SOM)
      || (insys == SPCS))
     {
     /* For projections no longer supported via the old interface, consider it
        an error */
     *iflg = ERROR;
     return;
     }
  else
  if (insys == ALBERS)
     {
     /* this is the call to initialize ALBERS 
     ---------------------------------------*/
     lat1 = paksz(inparm[2],iflg) * 3600 * S2R;
     if (*iflg != 0)
         return;
     lat2 = paksz(inparm[3],iflg) * 3600 * S2R;
     if (*iflg != 0)
         return;
     center_long = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     lat_origin  = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = alberinvint(r_major,r_minor,lat1,lat2,center_long,lat_origin,
			false_easting, false_northing);
     inv_trans[insys] = alberinv;
     }
  else
  if (insys == MERCAT)
     {
     /* this is the call to initialize MERCATOR
     ----------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     lat1   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = merinvint(r_major,r_minor,center_long,lat1,false_easting,
                     false_northing);
     inv_trans[insys] = merinv;
     }
  else
    if (insys == EQUIDC)
    {
     /* this is the call to initialize EQUIDISTANT CONIC
     ---------------------------------------------------*/
    lat1 = paksz(inparm[2],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    lat2 = paksz(inparm[3],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    lat_origin   = paksz(inparm[5],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    if (inparm[8] == 0)
       mode = 0;
    else
       mode = 1;
    *iflg = eqconinvint(r_major,r_minor,lat1,lat2,center_long,lat_origin,
		false_easting,false_northing,mode);
    inv_trans[insys] = eqconinv;
    }
  else
  if (insys == STEREO)
     {
     /* this is the call to initialize STEREOGRAPHIC
     ---------------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = sterinvint(radius,center_long,center_lat,false_easting, 
		       false_northing); 
     inv_trans[insys] = sterinv;
     }
  else
  if (insys == LAMAZ)
     {
     /* this is the call to initialize LAMBERT AZIMUTHAL EQUAL-AREA 
     -------------------------------------------------------------*/
     center_long = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat  = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = lamazinvint(radius, center_long, center_lat,false_easting,
			false_northing);
     inv_trans[insys] = lamazinv;
     }
  else
  if (insys == AZMEQD)
     {
     /* this is the call to initialize AZIMUTHAL EQUIDISTANT
     ------------------------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = aziminvint(radius,center_long,center_lat,false_easting,
		       false_northing); 
     inv_trans[insys] = aziminv;
     }
  else
  if (insys == GNOMON)
     {
     /* this is the call to initialize GNOMONIC 
     ----------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = gnominvint(radius,center_long,center_lat,false_easting,
                      false_northing);
     inv_trans[insys] = gnominv;
     }
  else
  if (insys == ORTHO)
     {
     /* this is the call to initialize ORTHOGRAPHIC
     --------------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = orthinvint(radius,center_long,center_lat,false_easting,
		       false_northing); 
     inv_trans[insys] = orthinv;
     }
  else
  if (insys == GVNSP)
     {
     /* this is the call to initialize GENERAL VERTICAL NEAR SIDED PERSPECTIVE 
     -----------------------------------------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     center_lat   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     h = inparm[2];
     *iflg = gvnspinvint(radius,h,center_long,center_lat,false_easting,
			false_northing);
     inv_trans[insys] = gvnspinv;
     }
  else
  if (insys == SNSOID)
     {
     /* this is the call to initialize SINUSOIDAL 
     --------------------------------------------*/
     center_long    = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = sininvint(radius, center_long,false_easting,false_northing);
     inv_trans[insys] = sininv;
     }
  else
  if (insys == EQRECT)
     {
     /* this is the call to initialize EQUIRECTANGULAR
     -----------------------------------------------*/
     center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     lat1   = paksz(inparm[5],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = equiinvint(radius,center_long,lat1,false_easting,
		       false_northing); 
     inv_trans[insys] = equiinv;
     }
  else
  if (insys == MILLER)
    {
    /* this is the call to initialize MILLER CYLINDRICAL
    --------------------------------------------------*/
    center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    *iflg = millinvint(radius, center_long,false_easting,false_northing);
    inv_trans[insys] = millinv;
    }
  else
  if (insys == VGRINT)
    {
    /* this is the call to initialize VAN DER GRINTEN 
    -----------------------------------------------*/
    center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    *iflg = vandginvint(radius, center_long,false_easting,false_northing);
    inv_trans[insys] = vandginv;
    }
  else
  if (insys == HAMMER)
    {
    /* this is the call to initialize HAMMER 
    --------------------------------------*/
    center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    *iflg = haminvint(radius, center_long,false_easting,false_northing);
    inv_trans[insys] = haminv;
    }
  else
  if (insys == ROBIN)
    {
    /* this is the call to initialize ROBINSON 
    ----------------------------------------*/
    center_long  = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    *iflg = robinvint(radius, center_long,false_easting,false_northing);
    inv_trans[insys] = robinv;
    }
  else
  if (insys == GOOD)
     {
     /* this is the call to initialize GOODE'S HOMOLOSINE
     ---------------------------------------------------*/
     *iflg = goodinvint(radius);
     inv_trans[insys] = goodinv;
     }
  else
  if (insys == MOLL)
     {
     /* this is the call to initialize MOLLWEIDE
     -------------------------------------------*/
     center_long = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = molwinvint(radius, center_long,false_easting,false_northing);
     inv_trans[insys] = molwinv;
     }
  else
  if (insys == IMOLL)
     {
     /* this is the call to initialize INTERRUPTED MOLLWEIDE 
     -----------------------------------------------------*/
     *iflg = imolwinvint(radius);
     inv_trans[insys] = imolwinv;
     }
  else
  if (insys == ALASKA)
     {
     /* this is the call to initialize ALASKA CONFORMAL 
     ------------------------------------------------*/
     *iflg = alconinvint(r_major,r_minor,false_easting,false_northing);
     inv_trans[insys] = alconinv;
     }
  else
  if (insys == WAGIV)
     {
     /* this is the call to initialize WAGNER IV 
     -----------------------------------------*/
     center_long = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = wivinvint(radius, center_long,false_easting,false_northing);
     inv_trans[insys] = wivinv;
     }
  else
  if (insys == WAGVII)
     {
     /* this is the call to initialize WAGNER VII 
     ------------------------------------------*/
     center_long = paksz(inparm[4],iflg) * 3600 * S2R;
     if (*iflg != 0)
        return;
     *iflg = wviiinvint(radius, center_long,false_easting,false_northing);
     inv_trans[insys] = wviiinv;
     }
  else
  if (insys == OBEQA)
    {
    /* this is the call to initialize OBLATED EQUAL AREA
    ---------------------------------------------------*/
    center_long = paksz(inparm[4],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    center_lat  = paksz(inparm[5],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    shape_m = inparm[2];
    shape_n = inparm[3];
    angle = paksz(inparm[8],iflg) * 3600 * S2R;
    if (*iflg != 0)
       return;
    *iflg = obleqinvint(radius,center_long,center_lat,shape_m, shape_n,
                angle,false_easting,false_northing);
    inv_trans[insys] = obleqinv;
    }
  else if (insys == ISIN)
    {
    /* this is the call to initialize INTEGERIZED SINUSOIDAL GRID
    ------------------------------------------------------------ */
    center_long = paksz( inparm[4], iflg ) * 3600 * S2R;
    if ( *iflg != 0 )
        return;
    dzone = inparm[8];
    djustify = inparm[10];

    *iflg = isinusinvinit( radius, center_long, false_easting, false_northing,
                           dzone, djustify );
    inv_trans[insys] = isinusinv;
    }
}
